//
// $Id: fd.cc,v 1.6 2001/07/13 15:15:06 nishi Exp $
//
// Copyright (C) 2001 Shouhei Nishi.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of author may not be used to endorse or promote products
//    derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//





#include "bochs.h"
#include "biosregs.h"
#define LOG_THIS bx_fd_bios.

#define FROM_FLOPPY 10
#define TO_FLOPPY   11

bx_fd_bios_c bx_fd_bios;
#if BX_USE_FDBIOS_SMF
#define this (&bx_fd_bios)
#endif


bx_fd_bios_c::bx_fd_bios_c(void)
{
}

bx_fd_bios_c::~bx_fd_bios_c(void)
{
  // nothing for now
}


  void
bx_fd_bios_c::init(bx_bios_c *b, bx_disk_bios_c *d)
{
  int i;

  BX_FD_BIOS_THIS bios = b;
  BX_FD_BIOS_THIS disk_bios = d;

  //
  // Floppy A setup
  //

  for(i=0; i<NUM_FLOPPY;i++) {
    BX_FD_BIOS_THIS s.media_present[i] = 0;
    BX_FD_BIOS_THIS drvmode1[i] = 0x03; /* 1024bytes/sect, 2HD, MFM */
    BX_FD_BIOS_THIS drvmode2[i] = 0x0208; /* 8 sectors, 2 heads */
  }

  BX_FD_BIOS_THIS s.media[0].bytes_per_sector  = 0;
  BX_FD_BIOS_THIS s.media[0].sectors_per_track = 0;
  BX_FD_BIOS_THIS s.media[0].tracks            = 0;
  BX_FD_BIOS_THIS s.media[0].heads             = 0;
  BX_FD_BIOS_THIS s.media[0].sectors           = 0;
  BX_FD_BIOS_THIS s.media[0].type              = BX_FLOPPY_NONE;
  BX_FD_BIOS_THIS s.media[0].fd = -1;
  BX_FD_BIOS_THIS s.media_present[0] = 0;


  if (bx_options.floppya.type != BX_FLOPPY_NONE) {
    BX_FD_BIOS_THIS s.num_supported_floppies++;
    if ( bx_options.floppya.initial_status == BX_INSERTED) {
      if (evaluate_media(bx_options.floppya.type, bx_options.floppya.path,
                   & BX_FD_BIOS_THIS s.media[0]))
        BX_FD_BIOS_THIS s.media_present[0] = 1;
      }
    }


  //
  // Floppy B setup
  //
  BX_FD_BIOS_THIS s.media[1].bytes_per_sector  = 0;
  BX_FD_BIOS_THIS s.media[1].sectors_per_track = 0;
  BX_FD_BIOS_THIS s.media[1].tracks            = 0;
  BX_FD_BIOS_THIS s.media[1].heads             = 0;
  BX_FD_BIOS_THIS s.media[1].sectors           = 0;
  BX_FD_BIOS_THIS s.media[1].type              = BX_FLOPPY_NONE;
  BX_FD_BIOS_THIS s.media[1].fd = -1;
  BX_FD_BIOS_THIS s.media_present[1] = 0;

  if (bx_options.floppyb.type != BX_FLOPPY_NONE) {
    BX_FD_BIOS_THIS s.num_supported_floppies++;
    if ( bx_options.floppyb.initial_status == BX_INSERTED) {
      if (evaluate_media(bx_options.floppyb.type, bx_options.floppyb.path,
                   & BX_FD_BIOS_THIS s.media[1]))
        BX_FD_BIOS_THIS s.media_present[1] = 1;
      }
    }
}

  void
bx_fd_bios_c::bios_exec(BX_CPU_C *cpu,
			int type,
			Bit32u addr,
			Bit32u errorno,
			Boolean isPM,
			int &exittype,
			Bit32u &exitaddr,
			Bit16u &exitseg,
			Bit64u &leasttime,
			Bit64u &maxtime)
{
  int drvno;
  vaddrinfo vs,vd;
  Bit8u buf[1024];
  int i;
  Bit32u offset;

  drvno = AL & 0x0f;

  BX_INFO(("FD BIOS: call %d %.8x %d:\n",type,addr,isPM));
  cpu->atexit();

  if(drvno >= NUM_FLOPPY) {
    AH = 0x02;
    goto done;
  }

  switch(AH) {
  case 0x00:
    BX_FD_BIOS_THIS drvmode1[drvno] = (DX & 0xff);
    BX_FD_BIOS_THIS drvmode2[drvno] = BX;
    goto OK;
  case 0x01:
    DX = (DX & 0xff00) | (BX_FD_BIOS_THIS drvmode1[drvno] & 0xff);
    BX = BX_FD_BIOS_THIS drvmode2[drvno];
    goto OK;
  case 0x02:
    DX &= 0xff00;
    if(!BX_FD_BIOS_THIS s.media_present[drvno]) DX |= 0x01;
    if(BX_FD_BIOS_THIS s.media[drvno].heads >1) DX |= 0x04;
    switch (BX_FD_BIOS_THIS s.media[drvno].type) {
      case BX_FLOPPY_720K: // 720K 3.5"
	DX |= 0x10;
	break;
      case BX_FLOPPY_1_2: // 1.2M 5.25"
      case BX_FLOPPY_1_44: // 1.44M 3.5"
      case BX_FLOPPY_2_88: // 2.88M 3.5"
      case BX_FLOPPY_1_2J: // 1.2M(PC98)
	break;
    }
    BX_FD_BIOS_THIS drvmode1[drvno] = 
      (DX & 0xf0) | (BX_FD_BIOS_THIS drvmode1[drvno] & 0x0f);
    if(!BX_FD_BIOS_THIS s.media_present[drvno]) goto notready;
    goto OK;
  case 0x03:
    goto OK;
  case 0x05:
  case 0x06:
  case 0x07:
    if(!BX_FD_BIOS_THIS s.media_present[drvno]) goto notready;
    if(((BX_FD_BIOS_THIS drvmode1[drvno] & 0x03) == 0x02 &&
       BX_FD_BIOS_THIS s.media[drvno].bytes_per_sector != 512) ||
       ((BX_FD_BIOS_THIS drvmode1[drvno] & 0x03) == 0x03 &&
	BX_FD_BIOS_THIS s.media[drvno].bytes_per_sector != 1024) ||
       CX < 0 ||
       CX >= BX_FD_BIOS_THIS s.media[0].tracks ||
       (DX >> 8) < 0 ||
       (DX >> 8) >= BX_FD_BIOS_THIS s.media[drvno].heads ||
       (DX & 0xff) < 1 ||
       (DX & 0xff) > BX_FD_BIOS_THIS s.media[drvno].sectors_per_track) goto notfound;
    if(AH == 0x07) goto OK;
    offset = ((CX * BX_FD_BIOS_THIS s.media[drvno].heads
	       + (DX >> 8)) * BX_FD_BIOS_THIS s.media[drvno].sectors_per_track
	      + ((DX & 0xff)-1)) * BX_FD_BIOS_THIS s.media[drvno].bytes_per_sector;
    for(i=0;i<BX;i++) {
      if(AH == 0x05) {
	floppy_xfer(drvno,
		    offset + BX_FD_BIOS_THIS s.media[drvno].bytes_per_sector * i,
		    buf,
		    BX_FD_BIOS_THIS s.media[drvno].bytes_per_sector,
		    FROM_FLOPPY);
	vs.vmem=0;
	vs.a.p=buf;
	vd.vmem=1;
	vd.a.v.s=DS;
	vd.a.v.o=DI+i*BX_FD_BIOS_THIS s.media[drvno].bytes_per_sector;
      } else {
	vs.vmem=1;
	vs.a.v.s=DS;
	vs.a.v.o=DI+i*BX_FD_BIOS_THIS s.media[drvno].bytes_per_sector;
	vd.vmem=0;
	vd.a.p=buf;
      }
      cpu->block_copy_virtual(vd,
			      vs,
			      cpu->sregs[BX_SEG_REG_CS].selector.rpl == 3,
			      BX_FD_BIOS_THIS s.media[drvno].bytes_per_sector,
			      1);
      if(AH == 0x06) {
	floppy_xfer(drvno,
		    offset + BX_FD_BIOS_THIS s.media[drvno].bytes_per_sector * i,
		    buf,
		    BX_FD_BIOS_THIS s.media[drvno].bytes_per_sector,
		    TO_FLOPPY);
      }
    }
    goto OK;
  case 0x08:
    goto OK;
  case 0x09:
    if(!BX_FD_BIOS_THIS s.media_present[drvno]) goto notready;
    buf[0] = CX;
    buf[1] = (DX >> 8);
    buf[2] = 1;
    buf[3] = (BX_FD_BIOS_THIS s.media[drvno].bytes_per_sector == 1024) ?
      3 : 2;
    vs.vmem=0;
    vs.a.p=buf;
    vd.vmem=1;
    vd.a.v.s=DS;
    vd.a.v.o=DI;
    cpu->block_copy_virtual(vd,
			    vs,
			    cpu->sregs[BX_SEG_REG_CS].selector.rpl == 3,
			    6,
			    0);
    goto OK;
  case 0x0e:
    goto OK;
  case 0xfb:
    goto OK;
  default:
    BX_PANIC(("FD BIOS: unsupported operation %.2x.\n",AH));
  }
notfound:
  CX = (CX & 0xff00) | 0x08;
  goto error;
notready:
  CX = (CX & 0xff00) | 0x01;
error:
  AH = 0x80;
  goto done;
OK:
  AH = 0x00;
done:
  cpu->set_CF(AH != 0x00);
}

  void
bx_fd_bios_c::floppy_xfer(Bit8u drive, Bit32u offset, Bit8u *buffer,
            Bit32u bytes, Bit8u direction)
{
  int ret;

  if (drive > 1)
    BX_PANIC(("floppy_xfer: drive > 1\n"));

  if (bx_dbg.floppy) {
    BX_INFO(("drive=%u\n", (unsigned) drive));
    BX_INFO(("offset=%u\n", (unsigned) offset));
    BX_INFO(("bytes=%u\n", (unsigned) bytes));
    BX_INFO(("direction=%s\n", (direction==FROM_FLOPPY)? "from" : "to"));
    }

#ifdef macintosh
  if (strcmp(bx_options.floppya.path, SuperDrive))
#endif
    {
    ret = lseek(BX_FD_BIOS_THIS s.media[drive].fd, offset, SEEK_SET);
    if (ret < 0) {
      BX_PANIC(("could not perform lseek() on floppy image file\n"));
      }
    }

  if (direction == FROM_FLOPPY) {
#ifdef macintosh
    if (!strcmp(bx_options.floppya.path, SuperDrive))
      ret = fd_read((char *) buffer, offset, bytes);
    else
#endif
      ret = ::read(BX_FD_BIOS_THIS s.media[drive].fd, (bx_ptr_t) buffer, bytes);
    if (ret < int(bytes)) {
      /* ??? */
      if (ret > 0) {
        BX_INFO(("floppy: partial read() on floppy image returns %u/%u\n",
          (unsigned) ret, (unsigned) bytes));
        memset(buffer + ret, 0, bytes - ret);
        }
      else {
        BX_INFO(("floppy: read() on floppy image returns 0\n"));
        memset(buffer, 0, bytes);
        }
      }
    }

  else { // TO_FLOPPY
#ifdef macintosh
    if (!strcmp(bx_options.floppya.path, SuperDrive))
      ret = fd_write((char *) buffer, offset, bytes);
    else
#endif
      ret = ::write(BX_FD_BIOS_THIS s.media[drive].fd, (bx_ptr_t) buffer, bytes);
    if (ret < int(bytes)) {
      BX_PANIC(("could not perform write() on floppy image file\n"));
      }
    }
}

  Boolean
bx_fd_bios_c::evaluate_media(unsigned type, char *path, floppy_t *media)
{
  struct stat stat_buf;
  int ret;
  char sTemp[1024];

  if (type == BX_FLOPPY_NONE)
    return(0);

  // open media file (image file or device)
#ifdef macintosh
  media->fd = 0;
  if (strcmp(bx_options.floppya.path, SuperDrive))
#endif
    media->fd = open(path, O_RDWR
#ifdef O_BINARY
                   | O_BINARY
#endif
                  );

  if (media->fd < 0) {
    //perror("floppy open of a:\n");
    sprintf(sTemp,"floppy open of %s:\n",path);
    perror(sTemp);
    return(0);
    }

#ifdef macintosh
  if (!strcmp(bx_options.floppya.path, SuperDrive))
    ret = fd_stat(&stat_buf);
  else
#endif
    ret = fstat(media->fd, &stat_buf);
  if (ret) {
    perror("fstat'ing floppy 0 drive image file");
    BX_PANIC(("fstat() returns error!\n"));
    return(0);
    }

  if ( S_ISREG(stat_buf.st_mode) ) {
    // regular file
    switch (type) {
      case BX_FLOPPY_720K: // 720K 3.5"
        media->type              = BX_FLOPPY_720K;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 9;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_1_2: // 1.2M 5.25"
        media->type              = BX_FLOPPY_1_2;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 15;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_1_44: // 1.44M 3.5"
        media->type              = BX_FLOPPY_1_44;
        if (stat_buf.st_size <= 1474560) {
	  media->bytes_per_sector  = 512;
          media->sectors_per_track = 18;
          media->tracks            = 80;
          media->heads             = 2;
          }
        else if (stat_buf.st_size == 1720320) {
	  media->bytes_per_sector  = 512;
          media->sectors_per_track = 21;
          media->tracks            = 80;
          media->heads             = 2;
          }
	else if (stat_buf.st_size == 1763328) {
	  media->bytes_per_sector  = 512;
          media->sectors_per_track = 21;
          media->tracks            = 82;
          media->heads             = 2;
	  }
        else {
          fprintf(stderr, "# floppy: evaluate_media: file '%s' of unknown size %lu\n",
            path, (unsigned long) stat_buf.st_size);
          BX_INFO(("floppy: evaluate_media: file '%s' of unknown size %lu\n",
            path, (unsigned long) stat_buf.st_size));
          return(0);
          }
        break;
      case BX_FLOPPY_2_88: // 2.88M 3.5"
        media->type              = BX_FLOPPY_2_88;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 36;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_1_2J: // 1.2M(PC98)
        media->type              = BX_FLOPPY_1_2J;
	media->bytes_per_sector  = 1024;
        media->sectors_per_track = 8;
        media->tracks            = 77;
        media->heads             = 2;
        break;
      default:
        BX_PANIC(("floppy: evaluate_media: unknown media type\n"));
      }
    media->sectors = media->heads * media->tracks * media->sectors_per_track;
    return(1); // success
    }

  else if ( S_ISCHR(stat_buf.st_mode)
#if BX_WITH_MACOS == 0
#ifdef S_ISBLK
            || S_ISBLK(stat_buf.st_mode)
#endif
#endif
           ) {
    // character or block device
    // assume media is formatted to typical geometry for drive
    switch (type) {
      case BX_FLOPPY_720K: // 720K 3.5"
        media->type              = BX_FLOPPY_720K;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 9;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_1_2: // 1.2M 5.25"
        media->type              = BX_FLOPPY_1_2;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 15;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_1_44: // 1.44M 3.5"
        media->type              = BX_FLOPPY_1_44;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 18;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_2_88: // 2.88M 3.5"
        media->type              = BX_FLOPPY_2_88;
	media->bytes_per_sector  = 512;
        media->sectors_per_track = 36;
        media->tracks            = 80;
        media->heads             = 2;
        break;
      case BX_FLOPPY_1_2J: // 1.2M(PC98)
        media->type              = BX_FLOPPY_1_2J;
	media->bytes_per_sector  = 1024;
        media->sectors_per_track = 8;
        media->tracks            = 77;
        media->heads             = 2;
        break;
      default:
        BX_PANIC(("floppy: evaluate_media: unknown media type\n"));
      }
    media->sectors = media->heads * media->tracks * media->sectors_per_track;
    return(1); // success
    }
  else {
    // unknown file type
    fprintf(stderr, "# floppy: unknown mode type\n");
    BX_INFO(("floppy: unknown mode type\n"));
    return(0);
    }
}
